// ===== MOBILE MENU TOGGLE =====
const mobileMenu = document.getElementById('mobile-menu');
const navMenu = document.querySelector('.nav-menu');

if (mobileMenu) {
    mobileMenu.addEventListener('click', () => {
        mobileMenu.classList.toggle('active');
        navMenu.classList.toggle('active');
    });

    // Close menu when clicking on a link
    document.querySelectorAll('.nav-menu li a').forEach(link => {
        link.addEventListener('click', () => {
            mobileMenu.classList.remove('active');
            navMenu.classList.remove('active');
        });
    });

    // Close menu when clicking outside
    document.addEventListener('click', (e) => {
        if (!mobileMenu.contains(e.target) && !navMenu.contains(e.target)) {
            mobileMenu.classList.remove('active');
            navMenu.classList.remove('active');
        }
    });
}

// ===== HERO SLIDER =====
let currentSlideIndex = 0;
const slides = document.querySelectorAll('.slide');
const indicators = document.querySelectorAll('.indicator');

function showSlide(index) {
    if (!slides.length) return;

    if (index >= slides.length) {
        currentSlideIndex = 0;
    } else if (index < 0) {
        currentSlideIndex = slides.length - 1;
    } else {
        currentSlideIndex = index;
    }

    slides.forEach((slide, i) => {
        slide.classList.remove('active-slide');
        if (indicators[i]) {
            indicators[i].classList.remove('active-indicator');
        }
    });

    slides[currentSlideIndex].classList.add('active-slide');
    if (indicators[currentSlideIndex]) {
        indicators[currentSlideIndex].classList.add('active-indicator');
    }
}

function changeSlide(direction) {
    showSlide(currentSlideIndex + direction);
}

function goToSlide(index) {
    showSlide(index);
}

// Auto slide
if (slides.length > 0) {
    showSlide(currentSlideIndex);
    setInterval(() => {
        changeSlide(1);
    }, 5000);
}

// ===== PORTFOLIO FILTER =====
const filterButtons = document.querySelectorAll('.filter-button');
const portfolioBoxes = document.querySelectorAll('.portfolio-box');

filterButtons.forEach(button => {
    button.addEventListener('click', () => {
        // Remove active class from all buttons
        filterButtons.forEach(btn => btn.classList.remove('active'));
        // Add active class to clicked button
        button.classList.add('active');

        const filterValue = button.getAttribute('data-filter');

        portfolioBoxes.forEach(box => {
            const category = box.getAttribute('data-category');

            if (filterValue === 'all' || category === filterValue) {
                box.style.display = 'block';
                setTimeout(() => {
                    box.style.opacity = '1';
                    box.style.transform = 'scale(1)';
                }, 10);
            } else {
                box.style.opacity = '0';
                box.style.transform = 'scale(0.8)';
                setTimeout(() => {
                    box.style.display = 'none';
                }, 300);
            }
        });
    });
});

// ===== PORTFOLIO MODAL =====
const modal = document.getElementById('portfolioModal');
const modalImage = document.getElementById('modalImage');
const modalCaption = document.getElementById('modalCaption');
let currentModalIndex = 0;
const allPortfolioImages = Array.from(document.querySelectorAll('.portfolio-box img'));

function openModal(index) {
    if (!modal || !modalImage) return;

    currentModalIndex = index;
    modal.classList.add('active');
    modalImage.src = allPortfolioImages[currentModalIndex].src;
    modalCaption.textContent = allPortfolioImages[currentModalIndex].alt;
    document.body.style.overflow = 'hidden';
}

function closeModal() {
    if (!modal) return;

    modal.classList.remove('active');
    document.body.style.overflow = 'auto';
}

function changeModalImage(direction) {
    currentModalIndex += direction;

    if (currentModalIndex >= allPortfolioImages.length) {
        currentModalIndex = 0;
    } else if (currentModalIndex < 0) {
        currentModalIndex = allPortfolioImages.length - 1;
    }

    modalImage.src = allPortfolioImages[currentModalIndex].src;
    modalCaption.textContent = allPortfolioImages[currentModalIndex].alt;
}

// Close modal when clicking outside image
if (modal) {
    modal.addEventListener('click', (e) => {
        if (e.target === modal) {
            closeModal();
        }
    });
}

// Keyboard navigation for modal
document.addEventListener('keydown', (e) => {
    if (!modal || !modal.classList.contains('active')) return;

    if (e.key === 'Escape') {
        closeModal();
    } else if (e.key === 'ArrowLeft') {
        changeModalImage(-1);
    } else if (e.key === 'ArrowRight') {
        changeModalImage(1);
    }
});

// ===== TESTIMONIALS SLIDER =====
let currentTestimonialIndex = 0;
const testimonialSlides = document.querySelectorAll('.testimonial-slide');

function showTestimonial(index) {
    if (!testimonialSlides.length) return;

    if (index >= testimonialSlides.length) {
        currentTestimonialIndex = 0;
    } else if (index < 0) {
        currentTestimonialIndex = testimonialSlides.length - 1;
    } else {
        currentTestimonialIndex = index;
    }

    testimonialSlides.forEach(slide => {
        slide.classList.remove('active-testimonial');
    });

    testimonialSlides[currentTestimonialIndex].classList.add('active-testimonial');
}

function changeTestimonial(direction) {
    showTestimonial(currentTestimonialIndex + direction);
}

// Auto change testimonial
if (testimonialSlides.length > 0) {
    showTestimonial(currentTestimonialIndex);
    setInterval(() => {
        changeTestimonial(1);
    }, 6000);
}

// ===== FAQ ACCORDION =====
const faqItems = document.querySelectorAll('.faq-item');

faqItems.forEach(item => {
    const question = item.querySelector('.faq-question');

    question.addEventListener('click', () => {
        // Close other items
        faqItems.forEach(otherItem => {
            if (otherItem !== item) {
                otherItem.classList.remove('active');
            }
        });

        // Toggle current item
        item.classList.toggle('active');
    });
});

// ===== CONTACT FORM =====
const contactForm = document.getElementById('contactForm');

if (contactForm) {
    contactForm.addEventListener('submit', (e) => {
        e.preventDefault();

        // Get form data
        const formData = new FormData(contactForm);
        const data = {};
        formData.forEach((value, key) => {
            data[key] = value;
        });

        // Log form data (in production, send to server)
        console.log('Form Data:', data);

        // Show success message
        alert('Thank you for contacting us! We will get back to you within 24 hours.');

        // Reset form
        contactForm.reset();
    });
}

// ===== COUNTER ANIMATION =====
const counters = document.querySelectorAll('.stat-number');
const counterSpeed = 200;

const observerOptions = {
    threshold: 0.5
};

const counterObserver = new IntersectionObserver((entries) => {
    entries.forEach(entry => {
        if (entry.isIntersecting) {
            const counter = entry.target;
            const target = parseInt(counter.getAttribute('data-target'));
            const increment = target / counterSpeed;

            const updateCounter = () => {
                const current = parseInt(counter.textContent);

                if (current < target) {
                    counter.textContent = Math.ceil(current + increment);
                    setTimeout(updateCounter, 10);
                } else {
                    counter.textContent = target + '+';
                }
            };

            updateCounter();
            counterObserver.unobserve(counter);
        }
    });
}, observerOptions);

counters.forEach(counter => {
    counterObserver.observe(counter);
});

// ===== SMOOTH SCROLL =====
document.querySelectorAll('a[href^="#"]').forEach(anchor => {
    anchor.addEventListener('click', function (e) {
        const href = this.getAttribute('href');
        if (href !== '#' && href.length > 1) {
            e.preventDefault();
            const target = document.querySelector(href);
            if (target) {
                const offsetTop = target.offsetTop - 80;
                window.scrollTo({
                    top: offsetTop,
                    behavior: 'smooth'
                });
            }
        }
    });
});

// ===== SCROLL ANIMATIONS =====
const animateOnScrollOptions = {
    threshold: 0.15,
    rootMargin: '0px 0px -50px 0px'
};

const animateOnScrollObserver = new IntersectionObserver((entries) => {
    entries.forEach(entry => {
        if (entry.isIntersecting) {
            entry.target.style.opacity = '1';
            entry.target.style.transform = 'translateY(0)';
        }
    });
}, animateOnScrollOptions);

// Observe elements for animation
document.querySelectorAll('.service-card, .portfolio-item, .why-card, .team-card, .value-card').forEach(el => {
    el.style.opacity = '0';
    el.style.transform = 'translateY(30px)';
    el.style.transition = 'all 0.6s ease';
    animateOnScrollObserver.observe(el);
});

// ===== STICKY HEADER SCROLL =====
let lastScroll = 0;
const header = document.querySelector('header');

window.addEventListener('scroll', () => {
    const currentScroll = window.pageYOffset;

    if (currentScroll > 100) {
        header.style.boxShadow = '0 4px 12px rgba(0,0,0,0.1)';
    } else {
        header.style.boxShadow = '0 2px 8px rgba(0,0,0,0.1)';
    }

    lastScroll = currentScroll;
});

// ===== SET MIN DATE FOR EVENT DATE INPUT =====
const eventDateInput = document.getElementById('date');
if (eventDateInput) {
    const today = new Date().toISOString().split('T')[0];
    eventDateInput.setAttribute('min', today);
}

// ===== LOADING ANIMATION =====
window.addEventListener('load', () => {
    document.body.classList.add('loaded');
    console.log('PixelPro Studio - Website Loaded Successfully! 📸');
});

// ===== LAZY LOADING FOR IMAGES =====
if ('IntersectionObserver' in window) {
    const imageObserver = new IntersectionObserver((entries) => {
        entries.forEach(entry => {
            if (entry.isIntersecting) {
                const img = entry.target;
                if (img.dataset.src) {
                    img.src = img.dataset.src;
                    img.removeAttribute('data-src');
                }
                imageObserver.unobserve(img);
            }
        });
    });

    document.querySelectorAll('img[data-src]').forEach(img => {
        imageObserver.observe(img);
    });
}

// ===== BACK TO TOP BUTTON =====
const backToTopBtn = document.createElement('button');
backToTopBtn.innerHTML = '<i class="fas fa-arrow-up"></i>';
backToTopBtn.className = 'back-to-top';
backToTopBtn.style.cssText = `
    position: fixed;
    bottom: 30px;
    right: 30px;
    width: 50px;
    height: 50px;
    background: linear-gradient(135deg, #ff6b35 0%, #f7931e 100%);
    color: white;
    border: none;
    border-radius: 50%;
    cursor: pointer;
    display: none;
    align-items: center;
    justify-content: center;
    font-size: 1.2rem;
    box-shadow: 0 4px 16px rgba(0,0,0,0.12);
    z-index: 999;
    transition: all 0.3s ease;
`;

document.body.appendChild(backToTopBtn);

window.addEventListener('scroll', () => {
    if (window.pageYOffset > 300) {
        backToTopBtn.style.display = 'flex';
    } else {
        backToTopBtn.style.display = 'none';
    }
});

backToTopBtn.addEventListener('click', () => {
    window.scrollTo({
        top: 0,
        behavior: 'smooth'
    });
});

backToTopBtn.addEventListener('mouseenter', () => {
    backToTopBtn.style.transform = 'translateY(-5px)';
});

backToTopBtn.addEventListener('mouseleave', () => {
    backToTopBtn.style.transform = 'translateY(0)';
});

// ===== PREVENT CONTEXT MENU ON IMAGES (Optional) =====
// Uncomment if you want to protect images
/*
document.querySelectorAll('img').forEach(img => {
    img.addEventListener('contextmenu', (e) => {
        e.preventDefault();
        return false;
    });
});
*/

console.log('All scripts loaded successfully! 🎉');
